-- (c) Meta Platforms, Inc. and affiliates. Confidential and proprietary.

function table.contains(table, element)
    for _, value in pairs(table) do
        if value == element then
            return true
        end
    end
    return false
  end

-- Get the median of a table.
function table.median( t )
    local temp={}

    -- deep copy table so that when we sort it, the original is unchanged
    -- also weed out any non numbers
    for k,v in pairs(t) do
        if type(v) == 'number' then
        table.insert( temp, v )
        end
    end

    table.sort( temp )

    -- If we have an even number of table elements or odd.
    if math.fmod(#temp,2) == 0 then
        -- return mean value of middle two elements
        return ( temp[#temp/2] + temp[(#temp/2)+1] ) / 2
    else
        -- return middle element
        return temp[math.ceil(#temp/2)]
    end
end

function jsonStringToTable(json_str)
  local json_parsed = json_str:match("{(.-)}") -- Extracting the JSON-like content
  local result = {}
  for key, value in json_parsed:gmatch("%s*([^:%s]+)%s*:%s*([^,%s]+)%s*") do
      result[key] = value
  end
  return result
end

function tableToJSONString(inputTable)
  local json_str = "{ "
  for key, value in pairs(inputTable) do
      json_str = json_str .. key .. " : " .. tostring(value) .. ", "
  end
  -- Removing the trailing comma and adding closing brace
  json_str = json_str:gsub(",%s*$", "") .. " }"
  return json_str
end

function enableBGClear(treatmentParams)
  -- Parse the input string as a Lua table
  local parsedTable = jsonStringToTable(treatmentParams)
  parsedTable["\"clear_background\""] = true
  return tableToJSONString(parsedTable)
end

function enableBGClearAllTreatments()
  CHROMATIC_ABER_1 = enableBGClear(CHROMATIC_ABER_1)
  CHROMATIC_ABER_2 = enableBGClear(CHROMATIC_ABER_2)
  CHROMATIC_ABER = {CHROMATIC_ABER_1, CHROMATIC_ABER_2}
  TRANSFORM_1 = enableBGClear(TRANSFORM_1)
  TRANSFORM_2 = enableBGClear(TRANSFORM_2)
  ZOOMBLUR_1 = enableBGClear(ZOOMBLUR_1)
end

function getConfig()
    -- default config
    local config = {
        maxItemCount = 15,
        isVideoSegmentationEnabled = false,
        minVideoSegmentDuration = 30,
        seedValue = 0,
        shouldAllowDuplicatedMedia = true,
        shouldEnableSingleMediaLogic = false,
        shouldEnableBGClearing = false
    }

    for _, mediaEvent in ipairs(MediaEvents) do
        if mediaEvent:getSourceMedia() == "config" then
            for i = 0, mediaEvent:size() - 1, 1 do
                local startTime, weight, duration, tags = mediaEvent:getEvent(i)
                local param = tags[1]
                if param == "maxItemCount" then
                    config.maxItemCount = math.floor(weight)
                elseif param == "isVideoSegmentationEnabled" then
                    config.isVideoSegmentationEnabled = weight == 1.0
                elseif param == "minVideoSegmentDuration" then
                    config.minVideoSegmentDuration = math.floor(weight)
                elseif param == "seed" then
                    config.seedValue = math.floor(weight)
                elseif param == "shouldEnableSingleMediaLogic" then
                    config.shouldEnableSingleMediaLogic = weight == 1.0
                elseif param == "shouldEnableBGClearing" then
                    config.shouldEnableBGClearing = weight == 1.0
                end
            end
        end
    end

    return config
end

function getBeatInfo()
    local beatsTable = {}
    local prevStart = 0
    local beatDurations = {}
    local medianDuration = -1
    for _, mediaEvent in ipairs(MediaEvents) do
        if mediaEvent:getSourceMedia() == "config" then goto CONTINUE end
        local numEvents = mediaEvent:size()
        local onsetsAfter = 0
        for i = 0, numEvents - 1, 1 do
        -- check 'beat' tag in an event
        local start, weights, _, tags = mediaEvent:getEvent(i)
        if (table.contains(tags, "beat") and table.contains(tags, "onset") ) then
            -- save previous beat's onsets after the beat
            if #beatsTable ~= 0 then
                beatsTable[#beatsTable].onsetsAfter = onsetsAfter
            end
            -- save the new beat
            local beat = {
                startTime = start,
                weight = weights,
                onset = table.contains(tags, "onset"),
                onsetsBefore = onsetsAfter, -- onsets before beat are previous beat's onsets after the beat
                onsetsAfter = 0, -- This will be updated by the next beat.
            }
            table.insert(beatsTable, beat)
            onsetsAfter = 0
            -- save the beat duration
            local currDuration = start - prevStart
            table.insert(beatDurations, currDuration)
            prevStart = start
        else
            if #beatsTable ~= 0 then
                onsetsAfter = onsetsAfter + 1
            end
        end
        end
        ::CONTINUE::
    end
    if #beatDurations > 1 then
        medianDuration = table.median(beatDurations)
    end
    return beatsTable, medianDuration
end

CHROMATIC_ABER_1 = [[{"intensity": 0.001}]]
CHROMATIC_ABER_2 = [[{"intensity": 0.001, "angle": 90.0}]]
CHROMATIC_ABER = {CHROMATIC_ABER_1, CHROMATIC_ABER_2}

TRANSFORM_1 = [[
{
    "start_zoom": 1.0,
    "mid_zoom": 1.5,
    "global": false,
    "start_curve": "EASE_IN_QUAD"
}
]];

TRANSFORM_2 = [[
    {
        "start_zoom": 1.0,
        "mid_zoom": 1.1,
        "global": false,
        "start_curve": "TRIANGLE"
    }
]];

ZOOMBLUR_1 = [[
    {
        "start_strength": 0.0,
        "end_strength": 1.0,
        "curve": "EASE_IN_QUAD"
    }
]];

function addClip(clipDuration, mediaIndex)
    local mediaItem = MediaItems[mediaIndex]
    local startTime = mediaItem:getStartSec()
    local speed = 1.0
    Project:addClip(startTime, clipDuration, speed, mediaItem)
end

function splitTimelineInParts(inputTimeline, percentages)
    local totalClips = #inputTimeline
    local outputTimeline = {}
    local startIndex = 1

    for i, percentage in ipairs(percentages) do
        local endIndex = startIndex + math.floor(totalClips * percentage)
        local partTable = {}

        for j = startIndex, endIndex do
            table.insert(partTable, inputTimeline[j])
        end

        table.insert(outputTimeline, partTable)
        startIndex = endIndex + 1
    end

    return outputTimeline
end

function addSingleMediaEffect(clipDuration)
    Treatments:chromaticAberrationTreatment(0.0, clipDuration/8, CHROMATIC_ABER[1])
    Treatments:transformTreatment(2*clipDuration/8, 3*clipDuration/8, TRANSFORM_2)
    Treatments:chromaticAberrationTreatment(2*clipDuration/8, 3*clipDuration/8, CHROMATIC_ABER[2])
    local transitionTime = 0.2
    local transitionStart =  4.5*clipDuration/8 - transitionTime
    Treatments:zoomBlurTreatment(transitionStart, 4.5*clipDuration/8, ZOOMBLUR_1)
    Treatments:transformTreatment(transitionStart, 4.5*clipDuration/8, TRANSFORM_1)
    transitionStart =  6*clipDuration/8 - transitionTime
    Treatments:zoomBlurTreatment(transitionStart, 6*clipDuration/8, ZOOMBLUR_1)
    Treatments:transformTreatment(transitionStart, 6*clipDuration/8, TRANSFORM_1)
    Treatments:chromaticAberrationTreatment(6*clipDuration/8, 7*clipDuration/8, CHROMATIC_ABER[1])
    Treatments:transformTreatment(7*clipDuration/8, 8*clipDuration/8, TRANSFORM_2)
    Treatments:chromaticAberrationTreatment(7*clipDuration/8, 8*clipDuration/8, CHROMATIC_ABER[2])
end

function addEffect(partIndex, clipIndex, clipDuration)
    if (partIndex == 1 or partIndex == 3) then
        local chromaticAberIndex = (clipIndex % 2) + 1
        Treatments:chromaticAberrationTreatment(0.0, clipDuration*0.5, CHROMATIC_ABER[chromaticAberIndex])

        if (clipIndex % 3 == 0) then
            Treatments:transformTreatment(0.0, clipDuration, TRANSFORM_2)
        end
    else
        local transitionTime = 0.2
        local transitionStart = 0.0
        if clipDuration > transitionTime then
            transitionStart = clipDuration - transitionTime
        end
        Treatments:zoomBlurTreatment(transitionStart, clipDuration, ZOOMBLUR_1)
        Treatments:transformTreatment(transitionStart, clipDuration, TRANSFORM_1)
    end
end

function main()
    -- Keep only beats that fit in the suggested timeline
    -- https://www.wolframalpha.com/input?i=0.02+*+%28x+%5E+2%29+-+0.6+*+x+%2B+6%2C+x+from+1.0+to+15&fbclid=IwAR28NqOXDY9gC2qRCoc5yoEzfyEdBjtZzgz4-XIP2Gw7rqxDu3r7oBg5etM
    local numMediaItems = #MediaItems
    local config = getConfig()
    if config.shouldEnableBGClearing then
      enableBGClearAllTreatments()
    end
    local suggestedClipDuration = 0.02 * (numMediaItems ^ 2) - 0.6 * numMediaItems + 6
    local singleMedia = (numMediaItems == 1 and config.shouldEnableSingleMediaLogic)
    if (singleMedia) then
        suggestedClipDuration = 8.0
    end

    -- Find beatDuration
    local _, beatDuration = getBeatInfo()

    local clips = {}
    for i = 1, numMediaItems, 1 do
        -- Sync clip to the beat
        local clipDuration = suggestedClipDuration
        if (beatDuration > 0.0 and suggestedClipDuration > beatDuration) then
            local beatsCount = math.floor(suggestedClipDuration/beatDuration)
            clipDuration = beatsCount * beatDuration
        end

        local mediaItem = MediaItems[i]
        if (mediaItem:isVideo()) then
            local mediaDuration = mediaItem:getDurationSec()
            if (mediaDuration < clipDuration) then
                -- In this case ignore the beat
                clipDuration = mediaDuration
            end
        end

        local clip = {mediaIndex = i, duration = clipDuration}
        table.insert(clips, clip)
    end

    -- Split clips in areas to apply the effects
    local percentages = {0.15, 0.35, 0.5}
    local parts = splitTimelineInParts(clips, percentages)
    for i, part in ipairs(parts) do
        for j, clip in ipairs(part) do
            addClip(clip.duration, clip.mediaIndex)
            if (singleMedia) then
                addSingleMediaEffect(clip.duration)
            else
                addEffect(i, clip.mediaIndex, clip.duration)
            end
        end
    end
end

main()
