#version 300 es
precision highp float;

uniform sampler2D u_inputTexture;
uniform sampler2D u_inputMask;
uniform sampler2D u_inputSearchMask;
uniform sampler2D u_inpaintTexture;
uniform sampler2D u_targetToSource;
uniform sampler2D u_targetTexture;
uniform sampler2D uPropagationTexture;

uniform vec4 uResolution;
uniform vec2 uPatchSize;
uniform float u_level;
uniform float u_seed;
uniform int u_w;
uniform int uRenderMode; // 0: no seprate, 1: propagation, 2: random search
uniform bool u_isFirst;

in vec2 v_TextureCoordinates;
layout(location=0) out vec4 out_targetToSource;


float random(vec2 uv)
{
    return fract(sin(dot(uv, vec2(12.9898, 78.233))) * u_seed);
}

bool isMask(sampler2D mask, vec2 texcoor, float level)
{
    return textureLod(mask, texcoor, level).a > 0.99;
}

bool isAlphaMask(sampler2D mask, vec2 texcoor, float level)
{
    return textureLod(mask, texcoor, level).a > 0.0;
}

bool needSkip(vec2 coor_ks, vec2 coor_kt, float level)
{
    if (coor_ks.x < 0.0 || coor_ks.x > 1.0) {return true;}
    if (coor_ks.y < 0.0 || coor_ks.y > 1.0) {return true;}

    if (coor_kt.x < 0.0 || coor_kt.x > 1.0) {return true;}
    if (coor_kt.y < 0.0 || coor_kt.y > 1.0) {return true;}

    if (u_isFirst)
    {
        if (isMask(u_inputMask, coor_ks, level)) { return true; }
        if (isMask(u_inputMask, coor_kt, level)) { return true; }
    }
    else
    {
        if (isAlphaMask(u_targetTexture, coor_ks, level)) { return true; }
        if (isMask(u_inputMask, coor_ks, level) && isMask(u_inputSearchMask, coor_kt, level) ) { return true; }

        if (isMask(u_inputMask, coor_kt, level)) { return true; }
        if (isAlphaMask(u_targetTexture, coor_kt, level) && isMask(u_inputSearchMask, coor_ks, level) ) { return true; }
    }
    return false;
}

float pixelDistance(sampler2D srcTex, vec2 coord, vec2 r_coord, float level)
{
    float dis = 0.0, wsum = 0.0;
    ///ssd: Sum of Squared Difference: RGB_value + gradient_x + gradient_y
    float ssdmax = 3.0;

    for (float dy0 = -uPatchSize.y; dy0 <= uPatchSize.y; dy0 += 1.0)
    {
        for (float dx0 = -uPatchSize.x; dx0 <= uPatchSize.x; dx0 +=1.0)
        {
            vec2 d = vec2(dx0, dy0) * uResolution.zw;
            vec2 coor_ks = coord + d;
            vec2 coor_kt = r_coord + d;

            wsum += 1.0;

            if (needSkip(coor_ks, coor_kt, level))
            {
                dis += 1.0;
                continue;
            }

            vec3 s_value = textureLod(srcTex, coor_ks, level).rgb;
            vec3 t_value = textureLod(srcTex, coor_kt, level).rgb;
            vec3 ssd_value_vec = (s_value - t_value) * (s_value - t_value);
            float s_inpaint = textureLod(u_inpaintTexture, coor_ks, level).a;
            float t_inpaint = textureLod(u_inpaintTexture, coor_kt, level).a;
            float ssd_inpaint = (s_inpaint - t_inpaint) * (s_inpaint - t_inpaint) * 3.0;
            float ssd = mix(ssd_inpaint, dot(ssd_value_vec, vec3(1.0)), 0.75);

            dis += (ssd / ssdmax);
        }
    }

    float res = dis / wsum;
    return clamp(res, res, 1.0);
}

vec4 offset2Texture(vec2 offset, float dis)
{
    float dir = step(0.0, offset.x) * 0.25 + step(0.0, offset.y) * 0.5 + 0.25;
    return vec4(abs(offset), dis, dir);
}

vec4 texture2Offset(vec4 offsetColor)
{
    int index = int(round(offsetColor.w * 4.0 - 1.0)); // int(round((offsetColor.w - 0.25) * 4.0));

    // vec2 g_direction[4];
    // g_direction[0] = vec2(-1.0, -1.0);
    // g_direction[1] = vec2( 1.0, -1.0);
    // g_direction[2] = vec2(-1.0,  1.0);
    // g_direction[3] = vec2( 1.0,  1.0);
    // return vec4(offsetColor.xy * g_direction[index], offsetColor.z, 1.0);
    vec2 dir = vec2(
        float((index + 2) % 2 * 2 - 1),
        float(int(index >= 2) * 2 - 1)
    );
    return vec4(offsetColor.xy * dir, offsetColor.z, 1.0);
}

bool equalTof(float a, float b)
{
    return abs(a - b) <= 0.000001;
}

vec3 pixelPropagation(sampler2D offsetMap, vec3 offset, vec2 coord, sampler2D src, float dir, float level)
{
    vec2 step = uResolution.zw * dir;
    vec3 tmp = offset;

    vec2 oldoffset = offset.xy + coord;
    float dp0 = pixelDistance(src, coord, oldoffset, level);
    vec2 xp1 = coord + step; // vec2(coord.x + step.x, coord.y + step.y);
    float dp = 0.0;
    if (xp1.y >= 0.0 && xp1.y <= 1.0)
    {
        vec2 newoffset = texture2Offset(textureLod(offsetMap, xp1, level)).xy + coord;
        dp = pixelDistance(src, coord, newoffset, level);

        if (dp < dp0) {
            tmp = vec3(newoffset - coord, dp);
            dp0 = dp;
        }
    }

    vec2 xm1 = coord - step; // vec2(coord.x - step.x, coord.y - step.y);
    if (xm1.x >= 0.0 && xm1.x <= 1.0)
    {
        vec2 newoffset = texture2Offset(textureLod(offsetMap, xm1, level)).xy + coord;
        dp = pixelDistance(src, coord, newoffset, level);

        if (dp < dp0) {
            tmp = vec3(newoffset - coord, dp);
            dp0 = dp;
        }
    }

    vec2 ym1 = vec2(coord.x + step.x, coord.y - step.y);
    if (ym1.y >= 0.0 && ym1.y <= 1.0 ) {
        vec2 newoffset = texture2Offset(textureLod(offsetMap, ym1, level)).xy + coord;
        dp = pixelDistance(src, coord, newoffset, level);

        if (dp < dp0) {
            tmp = vec3(newoffset - coord, dp);
            dp0 = dp;
        }
    }

    vec2 yp1 = vec2(coord.x - step.x , coord.y + step.y);
    if (yp1.y >= 0.0 && yp1.y <= 1.0 ) {
        vec2 newoffset = texture2Offset(textureLod(offsetMap, yp1, level)).xy + coord;
        dp = pixelDistance(src, coord, newoffset, level);

        if (dp < dp0) {
            tmp = vec3(newoffset - coord, dp);
            dp0 = dp;
        }
    }

    return tmp;
}

vec3 pixelRandomSearch(vec3 offset, vec2 coord, sampler2D src, float level)
{
    int w = u_w;

    float offest_seed = u_seed / 1000.0;
    float uv_offset = step(v_TextureCoordinates.x + v_TextureCoordinates.y, 0.0) * offest_seed;
    vec2 input_coord = v_TextureCoordinates + uv_offset;

    vec2 r_coord;
    vec3 tem = offset;
    vec2 originTexCoord = coord * uResolution.xy;

    vec2 oldoffset = offset.xy + coord;
    float dp0 = pixelDistance(src, coord, oldoffset, level);

    while(w > 0)
    {
        r_coord.x = random(input_coord);
        // prevent divide by zero
        r_coord.y = random(1.0 / (input_coord + r_coord.x));

        int x_offset = int(floor(r_coord.x * 10000.0)) % (2 * w) - w;
        int y_offset = int(floor(r_coord.y * 10000.0)) % (2 * w) - w;
        r_coord = clamp(vec2(x_offset, y_offset) * uResolution.zw + coord, 0.0, 1.0);

        float d = pixelDistance(src, coord, r_coord, level);
        if (d < dp0) {
            tem = vec3(r_coord - coord, d);
            dp0 = d;
        }

        w /= 2;
    }
    return tem;
}

bool containsMask(sampler2D mask, vec2 coord, float step, float level)
{
    for (float dx = -step; dx <= step; dx += 1.0)
    {
        for (float dy = -step; dy <= step; dy += 1.0)
        {
            vec2 newcoord = vec2(dx, dy) * uResolution.zw + coord;
            if (newcoord.x < 0.0 || newcoord.x > 1.0 || newcoord.y < 0.0 || newcoord.y > 1.0)
            {
                continue;
            }
            if (isMask(mask, newcoord, level))
            {
                return true;
            }
        }
    }
    return false;
}

void main()
{
    float searchSize = max(uPatchSize.x, uPatchSize.y);
    vec2 texCoord = v_TextureCoordinates;

    if (containsMask(u_inputMask, texCoord, searchSize, u_level))
    {
        vec3 targetToSource = texture2Offset(textureLod(u_targetToSource, texCoord, u_level)).rgb;
        if (equalTof(dot(targetToSource, vec3(1.0)), 0.0))
        {
            out_targetToSource = vec4(0.0, 0.0, 0.0, 1.0);
        }
        else
        {
            if (u_isFirst)
            {
                for (int d = 2; d >= 1; d /= 2)
                {
                    targetToSource = pixelPropagation(u_targetToSource, targetToSource, texCoord, u_inputTexture, float(d), u_level);
                }
                targetToSource = pixelRandomSearch(targetToSource, texCoord, u_inputTexture, u_level);
            }
            else
            {
                if (uRenderMode != 2)
                {
                    for (int d = 2; d >= 1; d /= 2)
                    {
                        targetToSource = pixelPropagation(u_targetToSource, targetToSource, texCoord, u_targetTexture, float(d), u_level);
                    }
                }
                else
                {
                    targetToSource = texture2Offset(textureLod(uPropagationTexture, texCoord, u_level)).rgb;
                }

                if (uRenderMode != 1)
                {
                    targetToSource = pixelRandomSearch(targetToSource, texCoord, u_targetTexture, u_level);
                }
            }
            out_targetToSource = offset2Texture(targetToSource.xy, targetToSource.z);
        }
    }
    else
    {
        out_targetToSource = vec4(0.0, 0.0, 0.0, 1.0);
    }
}
